import controlP5.*;
import java.io.File;
import processing.core.*;
import processing.pdf.*;
import processing.video.*;

ControlP5 cp5;
PGraphics textLayer, pointsLayer;
PImage selectedImage;
Movie video;

ArrayList<PVector> points = new ArrayList<>();
ArrayList<PVector[]> curves = new ArrayList<>();

PFont customFont, defaultFont;
String text = "FRU";

float sampleRate = 0.05;
int fontSize = 600;
float curveLength = 20;
float curveAmp = 22;
float numPoints = 10000; // Nombre de points

boolean typoChanged = false, useImage = false, useVideo = false, isSelectingImage = false;
boolean curvesChanged = false; // Flag pour vérifier si les courbes doivent être redessinées
boolean alwaysUpdate = false;

// Variables pour export multi-frame
boolean exportingPDFMulti = false;
boolean exportingPNGMulti = false;
int multiFrameCounter = 0;

// Dossiers pour les exports multi-frame
String multiPDFDir = "";
String multiPNGDir = "";

void preload() {
  defaultFont = createFont("FreeSans.ttf", fontSize);
  customFont = defaultFont;
  surfaceSettings();
}

void surfaceSettings() {
  surface.setLocation(0, 0);
  surface.setTitle("FRUFRU MAKER");
}

void setup() {
  size(1800, 1160);
    File exportDir = new File(sketchPath("_RENDUS"));
  if (!exportDir.exists()) {
    exportDir.mkdirs();
  }
  preload();
  createGUI();

  textLayer = createGraphics(width, height);
  pointsLayer = createGraphics(width, height);

  drawSource();
  generatePointsAndCurves();
}

void draw() {
  if (typoChanged) {
    typoChanged = false;
    generatePointsAndCurves();
  }

  background(240);
  
  if (alwaysUpdate) {
    if (frameCount % 3 == 0) { // Traiter toutes les 3 frames pour réduire la charge
      generatePointsAndCurves();
    }
  }

  if (useVideo) {
    if (frameCount % 3 == 0) { // Traiter toutes les 3 frames pour réduire la charge
      generatePointsAndCurves();
    }
  }

  displayCurves();
  image(pointsLayer, 0, 0);

  drawThumbnail();

  // Export multi-frame en PDF dans le dossier dédié
  if (exportingPDFMulti) {
    String fileName = multiPDFDir + "/" + nf(multiFrameCounter, 4) + ".pdf";
    PGraphics pdf = createGraphics(width, height, PDF, fileName);
    
    pdf.beginDraw();
    pdf.stroke(0);
    pdf.noFill();
    
    for (PVector[] curve : curves) {
      pdf.bezier(curve[0].x, curve[0].y,
                 curve[1].x, curve[1].y,
                 curve[2].x, curve[2].y,
                 curve[3].x, curve[3].y);
    }
    
    pdf.endDraw();
    pdf.dispose();
    
    println("PDF exporté : " + fileName);
    multiFrameCounter++;
  }

  // Export multi-frame en PNG dans le dossier dédié
  if (exportingPNGMulti) {
    String fileName = multiPNGDir + "/" + nf(multiFrameCounter, 4) + ".png";
    saveFrame(fileName);
    println("PNG exporté : " + fileName);
    multiFrameCounter++;
  }
}

void drawThumbnail() {
  push();
  textAlign(LEFT);
  translate(width - textLayer.width / 4 - 10, 10);
  image(textLayer, 0, 0, textLayer.width / 4, textLayer.height / 4);
  noFill();
  stroke(150);
  rect(0, 0, textLayer.width / 4, textLayer.height / 4);
  fill(150);
  noStroke();
  text("Source Layer", 10, 20);
  pop();
}

void createGUI() {
  cp5 = new ControlP5(this);
  int labelSize = 12, sliderWidth = 200, sliderHeight = 20;
  int posX = 20, posY = 20, step = 30;

  cp5.setColorActive(color(180, 0, 250))
     .setColorBackground(color(180))
     .setColorForeground(color(100))
     .setColorCaptionLabel(color(0))
     .setColorValueLabel(color(0));

  cp5.addTextfield("inputText")
     .setPosition(posX, posY)
     .setSize(sliderWidth, sliderHeight * 2)
     .setAutoClear(false)
     .setLabel("Enter text")
     .getCaptionLabel().setSize(labelSize);

  posY += 60;
  createButton("loadmyFont", posX, posY, sliderWidth, "Load Font", labelSize);
  posY += 40;
  createButton("loadImage", posX, posY, sliderWidth, "Load Image", labelSize);
  posY += 40;
  createButton("loadVideo", posX, posY, sliderWidth, "Load Video", labelSize);

  posY += step + 30;
  createSlider("sampleRate", posX, posY, sliderWidth, sliderHeight, 0, 1, sampleRate, "Sample Rate", labelSize);
  posY += step;
  createSlider("fontSize", posX, posY, sliderWidth, sliderHeight, 1, 1000, fontSize, "Font Size", labelSize);
  posY += step;
  createSlider("curveLength", posX, posY, sliderWidth, sliderHeight, 1, 200, curveLength, "Curve Length", labelSize);
  posY += step;
  createSlider("curveAmp", posX, posY, sliderWidth, sliderHeight, 1, 100, curveAmp, "Curve Amplitude", labelSize);
  posY += step;
  createSlider("numPoints", posX, posY, sliderWidth, sliderHeight, 1000, 100000, numPoints, "Number of Points", labelSize);

  posY += 40;
  createButton("exportPDF", posX, posY, sliderWidth, "Export to PDF", labelSize);
  posY += 40;
  
  // Boutons pour export multi-frame PDF
  createButton("startExportPDFMulti", posX, posY, sliderWidth, "Start Multi-frame PDF", labelSize);
  posY += 40;
  createButton("stopExportPDFMulti", posX, posY, sliderWidth, "Stop Multi-frame PDF", labelSize);
  posY += 40;
  
  // Bouton pour exporter une frame PNG unique
  createButton("exportPNG", posX, posY, sliderWidth, "Export Single Frame PNG", labelSize);
  posY += 40;
  
  // Boutons pour export multi-frame PNG
  createButton("startExportPNGMulti", posX, posY, sliderWidth, "Start Multi-frame PNG", labelSize);
  posY += 40;
  createButton("stopExportPNGMulti", posX, posY, sliderWidth, "Stop Multi-frame PNG", labelSize);
  posY += 40;

  createToggle("alwaysUpdate", posX, posY, "Update each frame", alwaysUpdate);
}

void createButton(String name, int x, int y, int width, String label, int labelSize) {
  cp5.addButton(name)
     .setPosition(x, y)
     .setSize(width, 30)
     .setLabel(label)
     .getCaptionLabel().setSize(labelSize);
}

void createSlider(String name, int x, int y, int w, int h, float min, float max, float val, String label, int labelSize) {
  cp5.addSlider(name)
     .setPosition(x, y)
     .setSize(w, h)
     .setRange(min, max)
     .setValue(val)
     .setLabel(label)
     .getCaptionLabel().setSize(labelSize);

  cp5.getController(name).addListener((ControlEvent event) -> typoChanged = true);
}

void createToggle(String name, int x, int y, String label, boolean def) {
  cp5.addToggle(name)
     .setPosition(x, y)
     .setSize(20, 20)
     .setValue(def)
     .setMode(ControlP5.SWITCH)
     .setLabel(label);
}

void inputText(String theText) {
  text = theText;
  useImage = useVideo = false;
  typoChanged = true;
}

void loadmyFont() {
  selectInput("Select a font file (.ttf, .otf):", "fileSelected");
}

void loadImage() {
  if (!isSelectingImage) {
    isSelectingImage = true;
    selectInput("Select an image file:", "imageSelected");
  }
}

void loadVideo() {
  selectInput("Select a video file:", "videoSelected");
}

void videoSelected(File selection) {
  if (selection != null) {
    video = new Movie(this, selection.getAbsolutePath());
    video.loop();
    useVideo = true;
    useImage = false;
    typoChanged = true;
  }
}

void fileSelected(File selection) {
  if (selection != null) {
    customFont = createFont(selection.getAbsolutePath(), fontSize);
    typoChanged = true;
  }
}

void imageSelected(File selection) {
  isSelectingImage = false;
  if (selection != null) {
    selectedImage = loadImage(selection.getAbsolutePath());
    useImage = selectedImage != null;
    useVideo = false;
    typoChanged = true;
  }
}

void drawSource() {
  textLayer.beginDraw();
  textLayer.background(255);
  textLayer.imageMode(CENTER);

  if (useVideo && video != null) {
    textLayer.image(video, width / 2, height / 2);
  } else if (useImage && selectedImage != null) {
    textLayer.image(selectedImage, width / 2, height / 2, width, height);
  } else {
    textLayer.fill(0);
    textLayer.textFont(customFont);
    textLayer.textSize(fontSize);
    textLayer.textAlign(CENTER, CENTER);
    textLayer.text(text, width / 2, height / 2);
  }

  textLayer.endDraw();
}

void generatePointsAndCurves() {
  drawSource(); // Mise à jour de la source
  points.clear();
  curves.clear();
  textLayer.loadPixels();

  int numPointsInt = int(numPoints);

  for (int i = 0; i < numPointsInt; i++) {
    int x = int(random(width));
    int y = int(random(height));
    int index = x + y * width;
    float b = brightness(textLayer.pixels[index]);
    if (random(1) < map(b, 0, 200, sampleRate, 0)) {
      points.add(new PVector(x, y));
      curves.add(generateCurve(x, y));
    }
  }

  curvesChanged = true;
}

PVector[] generateCurve(float x, float y) {
  float endX = x + random(-curveLength, curveLength);
  float endY = y + random(-curveLength, curveLength);

  return new PVector[] {
    new PVector(x, y),
    new PVector(x + random(-curveAmp, curveAmp), y + random(-curveAmp, curveAmp)),
    new PVector(endX + random(-curveAmp, curveAmp), endY + random(-curveAmp, curveAmp)),
    new PVector(endX, endY)
  };
}

void displayCurves() {
  if (!curvesChanged) return;

  pointsLayer.beginDraw();
  pointsLayer.background(240);
  pointsLayer.stroke(0);
  pointsLayer.noFill();
  
  for (PVector[] curve : curves) {
    pointsLayer.bezier(curve[0].x, curve[0].y,
                         curve[1].x, curve[1].y,
                         curve[2].x, curve[2].y,
                         curve[3].x, curve[3].y);
  }
  
  pointsLayer.endDraw();
  curvesChanged = false;
}

void exportPDF() {
  // Export single frame PDF avec nom unique
  String fileName = "_RENDUS/"+year() + "-" + nf(month(),2) + "-" + nf(day(),2) + "_" + nf(hour(),2) + "-" + nf(minute(),2) + "-" + nf(second(),2) + "-" + text + ".pdf";
  PGraphics pdf = createGraphics(width, height, PDF, fileName);
  
  pdf.beginDraw();
  pdf.stroke(0);
  pdf.noFill();
  
  for (PVector[] curve : curves) {
    pdf.bezier(curve[0].x, curve[0].y,
               curve[1].x, curve[1].y,
               curve[2].x, curve[2].y,
               curve[3].x, curve[3].y);
  }
  
  pdf.endDraw();
  pdf.dispose();
  println("PDF single exporté : " + fileName);
}

void startExportPDFMulti() {
  // Génère un nom de dossier unique en fonction de la date et l'heure
  multiPDFDir = "_RENDUS/export_pdf_" + year() + "-" + nf(month(),2) + "-" + nf(day(),2) + "_" + nf(hour(),2) + "-" + nf(minute(),2) + "-" + nf(second(),2);
  File dir = new File(multiPDFDir);
  if (!dir.exists()) {
    dir.mkdirs();
  }
  exportingPDFMulti = true;
  multiFrameCounter = 0;
  println("Export multi-frame PDF démarré dans le dossier : " + multiPDFDir);
}

void stopExportPDFMulti() {
  exportingPDFMulti = false;
  println("Export multi-frame PDF arrêté.");
}

void exportPNG() {
  // Export single frame PNG avec nom unique
  String fileName = "_RENDUS/export_" + year() + "-" + nf(month(),2) + "-" + nf(day(),2) + "_" + nf(hour(),2) + "-" + nf(minute(),2) + "-" + nf(second(),2) + ".png";
  saveFrame(fileName);
  println("PNG single exporté : " + fileName);
}

void startExportPNGMulti() {
  // Génère un nom de dossier unique en fonction de la date et l'heure
  multiPNGDir = "_RENDUS/export_png_" + year() + "-" + nf(month(),2) + "-" + nf(day(),2) + "_" + nf(hour(),2) + "-" + nf(minute(),2) + "-" + nf(second(),2);
  File dir = new File(multiPNGDir);
  if (!dir.exists()) {
    dir.mkdirs();
  }
  exportingPNGMulti = true;
  multiFrameCounter = 0;
  println("Export multi-frame PNG démarré dans le dossier : " + multiPNGDir);
}

void stopExportPNGMulti() {
  exportingPNGMulti = false;
  println("Export multi-frame PNG arrêté.");
}

void movieEvent(Movie m) {
  m.read();
}
